// General twochannel filterbank
// for constructing wavelet transforms
// (impules responses are read from file)

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangelista]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.

#include <stdio.h>
#include <string.h>
#include "m_pd.h"
#include "pd_ringbuffer.h"

#define SQRT2 1.4142135624
#define SQRT3 1.73205080757

static t_class *IWT_class;

typedef struct _IWT {
  t_object x_obj;
  
  t_rb_info *high_buffer;
  t_rb_info *low_buffer;
  t_rb_info *outbuffer;
  
  t_int     ir_len;
  t_float   *high_ir, *low_ir;  
} t_IWT;

void IWT_high(t_IWT *x, t_floatarg high_in)
{
  t_float low_in;
  if (x->low_buffer->is_empty==0) {
    low_in=airb_get(x->low_buffer);
    flrb_mult_add(x->outbuffer, x->high_ir, high_in);
    flrb_mult_add(x->outbuffer, x->low_ir,  low_in );
    outlet_float(x->x_obj.ob_outlet, flrb_get(x->outbuffer));
    flrb_put(x->outbuffer, 0.0);
    outlet_float(x->x_obj.ob_outlet, flrb_get(x->outbuffer));
    flrb_put(x->outbuffer, 0.0);
  } else {
    airb_put(x->high_buffer, high_in);
  }
}

void IWT_low(t_IWT *x, t_floatarg low_in)
{
  t_float high_in, out1, out2;
  if (x->high_buffer->is_empty==0) {
    high_in=airb_get(x->high_buffer);
    flrb_mult_add(x->outbuffer, x->high_ir, high_in);
    flrb_mult_add(x->outbuffer, x->low_ir,  low_in );
    out1=flrb_get(x->outbuffer);
    flrb_put(x->outbuffer, 0.0);
    out2=flrb_get(x->outbuffer);
    flrb_put(x->outbuffer, 0.0);
    outlet_float(x->x_obj.ob_outlet, out1);
    outlet_float(x->x_obj.ob_outlet, out2);
  } else {
    airb_put(x->low_buffer, low_in);
  }
}

// constructor
void *IWT_new(t_symbol *datafile)
{
  FILE  *fp;
  t_int fileok=0;
  short int ir_len=4;
  
  t_IWT *x=(t_IWT *)pd_new(IWT_class);
  
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("high"));
  
  outlet_new(&x->x_obj, &s_float);
  
  // open file if specified, otherwise use default IR
  if (datafile)
    if ((fp = fopen(datafile->s_name, "rb"))==NULL) {
      post("File %s does not exist",datafile->s_name);
      post("Using default 4-Tap Daubechies filter instead");
    } else {
      fread(&ir_len,1,2,fp);
      if (ir_len<0 || ir_len>20) {
        error("Impulse response length must be at least 1 and at most 20",datafile->s_name);
        post("Using default 4-Tap Daubechies filter instead");
        ir_len=4;
      } else {
        fileok=1;
      }
    }
    
  // initialise the (auto-increase) input buffers
  x->high_buffer = rb_create(1);
  x->low_buffer  = rb_create(1);
  
  x->ir_len=ir_len;
  
  // initialise the fixed-length output buffer
  x->outbuffer   = rb_create(ir_len);
  flrb_init(x->outbuffer);
  
  // set the impulse responses
  x->high_ir = getbytes(ir_len*sizeof(t_float));
  x->low_ir  = getbytes(ir_len*sizeof(t_float));
  
  // read impulse responses or put default IR
  if (fileok) {
    fread(x->high_ir,sizeof(t_float),ir_len,fp);
    fread(x->low_ir,sizeof(t_float),ir_len,fp);
  } else {
    x->low_ir[0] = SQRT2*(1-SQRT3)/8;
    x->low_ir[1] = SQRT2*(3-SQRT3)/8;
    x->low_ir[2] = SQRT2*(3+SQRT3)/8;
    x->low_ir[3] = SQRT2*(1+SQRT3)/8;

    x->high_ir[0]  =  x->low_ir[3];
    x->high_ir[1]  = -x->low_ir[2];
    x->high_ir[2]  =  x->low_ir[1];
    x->high_ir[3]  = -x->low_ir[0];
  }
  
  if (fp)
    fclose(fp);
  
  return (void *)x;
}

// destructor
void IWT_free(t_IWT *x) {
  // free the memory for impulse responses
  freebytes(x->high_ir, x->ir_len*sizeof(t_float));
  freebytes(x->low_ir,  x->ir_len*sizeof(t_float));
  
  // destroy the input and output buffers
  rb_destroy(x->high_buffer);
  rb_destroy(x->low_buffer);
  rb_destroy(x->outbuffer);
}

void IWT_setup(void) {
  IWT_class = class_new(gensym("IWT"),
                         (t_newmethod)IWT_new,
			 (t_method)IWT_free, 
                         sizeof(t_IWT),
			 CLASS_DEFAULT,
                         A_DEFSYMBOL, 0);
  class_addfloat(IWT_class, IWT_low);
  class_addmethod(IWT_class, (t_method)IWT_high, 
                  gensym("high"), A_DEFFLOAT, 0);
}

