// General inverse twochannel filterbank
// for constructing wavelet transforms
// (impules responses are read from file)

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangeli sta]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.


#include <stdio.h>
#include <string.h>
#include "m_pd.h"
#include "pd_ringbuffer.h"

#define SQRT2 1.4142135624
#define SQRT3 1.73205080757

static t_class *WT_class;

typedef struct _WT {
  t_object x_obj;
  
  t_rb_info *inbuffer;

  t_outlet  *high_out;
  t_outlet  *low_out;
  
  t_int     ir_len;
  t_int     even_sample;
  t_float   *high_ir, *low_ir;  
} t_WT;

void WT_float(t_WT *x, t_floatarg f)
{
  flrb_put(x->inbuffer, f);
  if (x->even_sample) {
    x->even_sample = 0;
    outlet_float(x->high_out, flrb_conv(x->inbuffer,x->high_ir));
    outlet_float(x->low_out,  flrb_conv(x->inbuffer,x->low_ir) );
  } else {
    x->even_sample = 1;
  }
}

// constructor
void *WT_new(t_symbol *datafile)
{
  FILE  *fp;
  t_int fileok=0;
  short int ir_len=4;
  
  t_WT *x=(t_WT *)pd_new(WT_class);
  
  x->low_out  = outlet_new(&x->x_obj, &s_float);
  x->high_out = outlet_new(&x->x_obj, &s_float);
  
  // open file if specified, otherwise use default IR
  if (datafile)
    if ((fp = fopen(datafile->s_name, "rb"))==NULL) {
      post("File %s does not exist",datafile->s_name);
      post("Using default 4-Tap Daubechies filter instead");
    } else {
      fread(&ir_len,1,2,fp);
      if (ir_len<0 || ir_len>20) {
        error("Impulse response length must be at least 1 and at most 20",datafile->s_name);
        post("Using default 4-Tap Daubechies filter instead");
        ir_len=4;
      } else {
        fileok=1;
      }
    }
      
  x->ir_len=ir_len;
  
  // initialise the fixed-length input buffer
  x->inbuffer   = rb_create(ir_len);
  flrb_init(x->inbuffer);
  
  // set the impulse responses
  x->high_ir = getbytes(ir_len*sizeof(t_float));
  x->low_ir  = getbytes(ir_len*sizeof(t_float));
  
  // read impulse responses or put default 4-tap Daubechies IR
  if (fileok) {
    fread(x->high_ir,sizeof(t_float),ir_len,fp);
    fread(x->low_ir,sizeof(t_float),ir_len,fp);
  } else {
    x->low_ir[0] = SQRT2*(1-SQRT3)/8;
    x->low_ir[1] = SQRT2*(3-SQRT3)/8;
    x->low_ir[2] = SQRT2*(3+SQRT3)/8;
    x->low_ir[3] = SQRT2*(1+SQRT3)/8;

    x->high_ir[0]  =  x->low_ir[3];
    x->high_ir[1]  = -x->low_ir[2];
    x->high_ir[2]  =  x->low_ir[1];
    x->high_ir[3]  = -x->low_ir[0];
  }
  
  if (fp)
    fclose(fp);
  
  return (void *)x;
}

// destructor
void WT_free(t_WT *x) {
  // free the memory for impulse responses
  freebytes(x->high_ir, x->ir_len*sizeof(t_float));
  freebytes(x->low_ir,  x->ir_len*sizeof(t_float));
  
  // destroy the input and output buffers
  rb_destroy(x->inbuffer);
}

void WT_setup(void) {
  WT_class = class_new(gensym("WT"),
                         (t_newmethod)WT_new,
			 (t_method)WT_free, 
                         sizeof(t_WT),
			 CLASS_DEFAULT,
                         A_DEFSYMBOL, 0);
  class_addfloat(WT_class, WT_float);
}

