// Grain generator, implementing one channel of an
// inverse cosine modulated filterbank

// Grain      = Impulse response of the filter
// Grain rate = sampling frequency / P 
// (P=filterbank order and also the downsampling factor)

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangelista]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.

#include "m_pd.h"
#include "pd_ringbuffer.h"
#include "math.h"

#define P_MAX 5000
#define P_MIN 64

// VC++ doesn't seem to like the single precision variants
// of sin, cos and fabs, so let's re#define them...
#ifdef NT
#define fabsf fabs
#define sinf  sin
#define cosf  cos
#endif
// ...neither does it provide the constant M_PI, so let's
// #define it ourselves
#ifdef NT
#define M_PI 3.14159
#endif

static t_class *icmf_tilde_class;

typedef struct _icmf_tilde {
  t_object  x_obj;
  t_sample  *iresp;
  t_sample  coef1,coef2;
  t_sample  *irptr1,*irptr2;
  t_int     index1,index2;
  t_rb_info *inbuffer;
  t_int     k,P;
  t_outlet  *b_out;
} t_icmf_tilde;

// write incoming coefficients into buffer
void icmf_tilde_coef(t_icmf_tilde *x, t_floatarg coef)
{
  airb_put(x->inbuffer, coef);
}

// modifiy impulse response if k has changed
void icmf_tilde_k(t_icmf_tilde *x, t_floatarg k)
{
  t_int    k_int     = (t_int)k;
  t_int    P_int     = x->P;
  t_int    n         = 0;
  t_int    irlen     = 2*P_int;
  t_sample *irptr    = x->iresp;
  t_float  pi_div_2P = M_PI/(2.0*P_int);
  
  x->k  = k_int;

  while (n<irlen) {
    *irptr++ = sinf(n*pi_div_2P)*
               cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P);
    n++;
  }
  
  x->irptr1 = 0;
  x->irptr2 = 0;
}

// generate new impulse response if P has changed
void icmf_tilde_P(t_icmf_tilde *x, t_floatarg P)
{
  t_int    k_int     = x->k;
  t_int    P_int     = x->P;
  t_int    n         = 0;
  t_int    irlen     = 2*P_int;
  t_sample *irptr;
  t_float  pi_div_2P;
  
  // maybe this should be done by the perform routine
  // once the old buffer is not used anymore
  freebytes(x->iresp, irlen*sizeof(t_sample));
  
  // calculate new ir length and pi/2P
  P_int     = (t_int)P;
  if (P_int<P_MIN)
    P_int=P_MIN;
  if (P_int>P_MAX)
    P_int=P_MAX;
  irlen     = 2*P_int;
  pi_div_2P = M_PI/(2.0*P_int);
  x->P  = P_int;

  // allocate memory for new impulse response
  irptr     = getbytes(irlen*sizeof(t_sample));
  x->iresp  = irptr;
  
  // write new impulse response
  while (n<irlen) {
    *irptr++ = sinf(n*pi_div_2P)*
               cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P);
    n++;
  }
  
  x->irptr1 = 0;
  x->irptr2 = 0;
}

t_int *icmf_tilde_perform(t_int *w)
{
  t_icmf_tilde *x = (t_icmf_tilde *)(w[1]);
  t_sample   *out =     (t_sample *)(w[2]);
  t_int    buflen =          (t_int)(w[3]);
  t_int         i,n=buflen;
  t_sample *outptr;
//  t_sample f_icmf = (x->f_icmf<0)?0.0:(x->f_icmf>1)?1.0:x->f_icmf;
  
//  while (n--) *out++ = (*in1++)*(1-f_icmf)+(*in2++)*f_icmf;
  if (x->irptr2==0)
  {
    if (x->irptr1==0)
    {
      if (airb_is_empty(x->inbuffer))
        // if no coefficients available, output zeros
        while (n--) *out++ = 0.0;
      else
      {
        // if a new coefficient is available, get it
        x->coef1 =airb_get(x->inbuffer);
        // setup impulse response pointer
        x->irptr1=x->iresp;
        // set the index to the value after calculating output
        x->index1=buflen;
        // generate the output
        while (n--) *out++ = x->coef1 * *(x->irptr1++);
      }
    }
    else
    {
      // calculate the output of the first grain
      // set index1
      x->index1+=buflen;
      // generate the output for grain 1
      while (n--) *out++ = x->coef1 * *(x->irptr1++);

      if (x->index1>x->P)
      {
        // if the second grain should be started, get
        // a coefficient (or 0 if none available)
        if (airb_is_empty(x->inbuffer))
          x->coef2=0.0;
        else
          x->coef2=airb_get(x->inbuffer);
        // setup impulse response
        x->irptr2=x->iresp;
        // set the index 2 to the value after calculation
        i=x->index1-x->P;
        x->index2=i;
        // calculate the output for grain 2
        out-=i;
        while (i--) *out++ += x->coef2 * *(x->irptr2++);
      }
    }
  }
  else
  {
    // remark: the case irptr1=0, irptr2<>0 cannot appear
    // so we have the 3 cases that grain 1 or grain 2 changes
    // or that none of them changes
    if (x->index1+buflen>2*x->P)
    {
      // case 1: grain 1 changes
      n=2*x->P-x->index1;
      i=buflen-n;
      while (n--) *out++ = x->coef1 * *(x->irptr1++) + 
                           x->coef2 * *(x->irptr2++);
      // get a coefficient (or 0 if none available)
      if (airb_is_empty(x->inbuffer))
        x->coef1=0.0;
      else
        x->coef1=airb_get(x->inbuffer);
      // setup impulse response and index for grain 1
      x->irptr1=x->iresp;
      x->index1=i;
      while (i--) *out++ = x->coef1 * *(x->irptr1++) + 
                           x->coef2 * *(x->irptr2++);
      x->index2+=buflen;
    }
    else
    {
      if (x->index2+buflen>2*x->P)
      {
        // case 2: grain 2 changes
        n=2*x->P-x->index2;
        i=buflen-n;
        while (n--) *out++ = x->coef1 * *(x->irptr1++) + 
                             x->coef2 * *(x->irptr2++);
        // get a coefficient (or 0 if none available)
        if (airb_is_empty(x->inbuffer))
          x->coef2=0.0;
        else
          x->coef2=airb_get(x->inbuffer);
        // setup impulse response and index for grain 1
        x->irptr2=x->iresp;
        x->index2=i;
        while (i--) *out++ = x->coef1 * *(x->irptr1++) + 
                             x->coef2 * *(x->irptr2++);
        x->index1+=buflen;
      }
      else
      {
        // case 3: no grain changes
        while (n--) *out++ = x->coef1 * *(x->irptr1++) + 
                             x->coef2 * *(x->irptr2++);
        x->index1+=buflen;
        x->index2+=buflen;
      }
    }
  }
    
  // if in the next buffer we need a new coefficient,
  // send a bang
  // simpler: if the buffer is empty, send a bang
  
  if (airb_is_empty(x->inbuffer))
    outlet_bang(x->b_out);
  
  return (w+4);
}

void icmf_tilde_dsp(t_icmf_tilde *x, t_signal **sp)
{
  dsp_add(icmf_tilde_perform, 3, x,
          sp[0]->s_vec,sp[0]->s_n);
}

// constructor
void *icmf_tilde_new(t_floatarg P, t_floatarg k)
{
  t_int        irlen,n,k_int,P_int;
  t_sample     *irptr;
  t_float      pi_div_2P;
  t_icmf_tilde *x = (t_icmf_tilde *)pd_new(icmf_tilde_class);
  
  // set P,k from constructor arguments
  P_int = (t_int)P;
  if (P_int<P_MIN)
    P_int=P_MIN;
  if (P_int>P_MAX)
    P_int=P_MAX;

  k_int = (t_int)k;
  
  x->P = P_int;
  x->k = k_int;

  x->irptr1  = 0;
  x->irptr2  = 0;
  
  // set up impulse response
  irlen    = 2*x->P;
  irptr    = getbytes(irlen*sizeof(t_sample));
  x->iresp = irptr;
  n=0;
  pi_div_2P=M_PI/(2.0*P_int);
  while (n<irlen) {
    *irptr++ = sinf(n*pi_div_2P)*
               cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P);
    n++;
  }
  
  // create input buffer
  x->inbuffer=rb_create(3);
  
  // create new inlets for P and k
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("P"));
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("k"));
  
  // signal outlet
  outlet_new(&x->x_obj, &s_signal);
  // coefficient request outlet
  x->b_out = outlet_new(&x->x_obj, &s_bang);  
  
  return (void *)x;
}

// destructor
void icmf_tilde_free(t_icmf_tilde *x)
{
  rb_destroy(x->inbuffer);
  freebytes(x->iresp,2*x->P*sizeof(t_sample));
}

void icmf_tilde_setup(void) {
  icmf_tilde_class = class_new(gensym("icmf~"),
        (t_newmethod)icmf_tilde_new,
	(t_method)icmf_tilde_free,
        sizeof(t_icmf_tilde),
	CLASS_DEFAULT,
	A_DEFFLOAT, A_DEFFLOAT, 0);
  
  class_addfloat(icmf_tilde_class, icmf_tilde_coef);
  class_addmethod(icmf_tilde_class, (t_method)icmf_tilde_P,
                  gensym("P"), A_DEFFLOAT, 0);
  class_addmethod(icmf_tilde_class, (t_method)icmf_tilde_k,
                  gensym("k"), A_DEFFLOAT, 0);
  class_addmethod(icmf_tilde_class,
        (t_method)icmf_tilde_dsp, gensym("dsp"), 0);
}
