// Grain generator, implementing one channel of an inverse
// cosine modulated filterbank and, additionally,
// grains which are distorted versions of the original
// impulse responses.

// see also module icmf

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangelista]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.

#include "m_pd.h"
#include "pd_ringbuffer.h"
#include "math.h"

#define P_MAX 5000
#define P_MIN 64

// VC++ doesn't seem to like the single precision variants
// of sin, cos and fabs, so let's re#define them...
#ifdef NT
#define fabsf fabs
#define sinf  sin
#define cosf  cos
#endif
// ...neither does it provide the constant M_PI, so let's
// #define it ourselves
#ifdef NT
#define M_PI 3.14159
#endif

static t_class *modgrain_tilde_class;

typedef struct _modgrain_tilde {
  t_object  x_obj;
  t_sample  *iresp,*iresp2,*iresp3;
  t_sample  coef1,coef2;
  t_sample  *ir1,*ir2;
  t_int     index1,index2;
  t_rb_info *inbuffer;
  t_int     k,P;
  t_float   alpha,gamma;
  t_outlet  *b_out;
} t_modgrain_tilde;

// write incoming coefficients into buffer
void modgrain_tilde_coef(t_modgrain_tilde *x, t_floatarg coef)
{
  airb_put(x->inbuffer, coef);
}

// some auxiliary functions for time and amplitude warping
inline t_float warp(t_float x, t_float alpha) {
  return(x*(1+alpha)/(1+alpha*fabsf(x)));
}

inline t_float timewarp(t_float x, t_float alpha) {
  return(x*(1+alpha)/(1+alpha*x/(t_float)M_PI));
}

t_float b2a(t_float b) {
  t_float b_norm=b;
  
  if (b_norm<-0.9999)
    b_norm=-0.9999;
  if (b_norm>0.9999)
    b_norm=0.9999;
  
  return(2/(b_norm+1)-2);
}

// modifiy impulse response if ampwarp factor has changed
void modgrain_tilde_ampwarp(t_modgrain_tilde *x, t_floatarg delta)
{
  t_int    k_int     = x->k;
  t_int    P_int     = x->P;
  t_int    n         = 0;
  t_int    irlen     = 2*P_int;
  t_sample *irptr,*irptrold;
  t_float  pi_div_2P = M_PI/(2.0*P_int);
  
  x->gamma=b2a(delta);
  
  // if iresp3 is not used, make it the new iresp 
  if ((x->ir1!=x->iresp3) && (x->ir2!=x->iresp3)) {
    irptr=x->iresp3;
  } else {
    // if iresp2 is not used, use that one
    if ((x->ir1!=x->iresp2) && (x->ir2!=x->iresp2)) {
      irptr=x->iresp2;
    } else {
      // otherwise it must be iresp
      irptr=x->iresp;
  }}
  
  irptrold=irptr;
  
  while (n<irlen) {
    *irptr++ = sinf(timewarp(n*pi_div_2P,x->gamma))*
               warp(cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P),x->alpha);
    n++;
  }
  
  // reorder iresp, iresp2 and iresp3 such that iresp=irptrold
  if (irptrold==x->iresp3) {
    x->iresp3 = x->iresp;
    x->iresp  = irptrold;
  } else if (irptrold==x->iresp2) {
    x->iresp2 = x->iresp;
    x->iresp  = irptrold;
  }
}

// modifiy impulse response if timewarp factor has changed
void modgrain_tilde_timewarp(t_modgrain_tilde *x, t_floatarg beta)
{
  t_int    k_int     = x->k;
  t_int    P_int     = x->P;
  t_int    n         = 0;
  t_int    irlen     = 2*P_int;
  t_sample *irptr,*irptrold;
  t_float  pi_div_2P = M_PI/(2.0*P_int);
  
  x->alpha=b2a(beta);
  
  // if iresp3 is not used, make it the new iresp 
  if ((x->ir1!=x->iresp3) && (x->ir2!=x->iresp3)) {
    irptr=x->iresp3;
  } else {
    // if iresp2 is not used, use that one
    if ((x->ir1!=x->iresp2) && (x->ir2!=x->iresp2)) {
      irptr=x->iresp2;
    } else {
      // otherwise it must be iresp
      irptr=x->iresp;
  }}
  
  irptrold=irptr;
  
  while (n<irlen) {
    *irptr++ = sinf(timewarp(n*pi_div_2P,x->gamma))*
               warp(cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P),x->alpha);
    n++;
  }
  
  // reorder iresp, iresp2 and iresp3 such that iresp=irptrold
  if (irptrold==x->iresp3) {
    x->iresp3 = x->iresp;
    x->iresp  = irptrold;
  } else if (irptrold==x->iresp2) {
    x->iresp2 = x->iresp;
    x->iresp  = irptrold;
  }
}

// modifiy impulse response if k has changed
void modgrain_tilde_k(t_modgrain_tilde *x, t_floatarg k)
{
  t_int    k_int     = (t_int)k;
  t_int    P_int     = x->P;
  t_int    n         = 0;
  t_int    irlen     = 2*P_int;
  t_sample *irptr,*irptrold;
  t_float  pi_div_2P = M_PI/(2.0*P_int);
  
  x->k  = k_int;
  
  // if iresp3 is not used, make it the new iresp 
  if ((x->ir1!=x->iresp3) && (x->ir2!=x->iresp3)) {
    irptr=x->iresp3;
  } else {
    // if iresp2 is not used, use that one
    if ((x->ir1!=x->iresp2) && (x->ir2!=x->iresp2)) {
      irptr=x->iresp2;
    } else {
      // otherwise it must be iresp
      irptr=x->iresp;
  }}
  
  irptrold=irptr;
  
  while (n<irlen) {
    *irptr++ = sinf(timewarp(n*pi_div_2P,x->gamma))*
               warp(cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P),x->alpha);
    n++;
  }
  
  // reorder iresp, iresp2 and iresp3 such that iresp=irptrold
  if (irptrold==x->iresp3) {
    x->iresp3 = x->iresp;
    x->iresp  = irptrold;
  } else if (irptrold==x->iresp2) {
    x->iresp2 = x->iresp;
    x->iresp  = irptrold;
  }
}

// generate new impulse response if P has changed
void modgrain_tilde_P(t_modgrain_tilde *x, t_floatarg P)
{
  t_int    k_int     = x->k;
  t_int    P_int     = x->P;
  t_int    n         = 0;
  t_int    irlen     = 2*P_int;
  t_sample *irptr;
  t_float  pi_div_2P;
  
  // release the memory used for the impulse responses
  freebytes(x->iresp, irlen*sizeof(t_sample));
  freebytes(x->iresp2, irlen*sizeof(t_sample));
  freebytes(x->iresp3, irlen*sizeof(t_sample));
  
  // calculate new ir length and pi/2P
  P_int     = (t_int)P;
  if (P_int<P_MIN)
    P_int=P_MIN;
  if (P_int>P_MAX)
    P_int=P_MAX;
  irlen     = 2*P_int;
  pi_div_2P = M_PI/(2.0*P_int);
  x->P  = P_int;

  // allocate memory for new impulse response
  // and the two backup buffers
  irptr     = getbytes(irlen*sizeof(t_sample));
  x->iresp  = irptr;
  x->iresp2 = getbytes(irlen*sizeof(t_sample));
  x->iresp3 = getbytes(irlen*sizeof(t_sample));
  
  // write new impulse response
  while (n<irlen) {
    *irptr++ = sinf(timewarp(n*pi_div_2P,x->gamma))*
               warp(cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P),x->alpha);
    n++;
  }
  
  x->ir1 = 0;
  x->ir2 = 0;
}

t_int *modgrain_tilde_perform(t_int *w)
{
  t_modgrain_tilde *x = (t_modgrain_tilde *)(w[1]);
  t_sample       *out =         (t_sample *)(w[2]);
  t_int        buflen =              (t_int)(w[3]);
  t_int           i,n = buflen;
  t_sample    *outptr, *irptr,*irptr2;

  if (x->ir2==0)
  {
    if (x->ir1==0)
    {
      if (airb_is_empty(x->inbuffer))
        // if no coefficients available, output zeros
        while (n--) *out++ = 0.0;
      else
      {
        // if a new coefficient is available, get it
        x->coef1 =airb_get(x->inbuffer);
        // setup impulse response pointer
        irptr=x->iresp;
        x->ir1=irptr;
        // set the index to the value after calculating output
        x->index1=buflen;
        // generate the output
        while (n--) *out++ = x->coef1 * *(irptr++);
      }
    }
    else
    {
      // calculate the output of the first grain
      // calculate irptr
      irptr = x->ir1+x->index1;
      // set index1
      x->index1+=buflen;
      // generate the output for grain 1
      while (n--) *out++ = x->coef1 * *(irptr++);

      if (x->index1>x->P)
      {
        // if the second grain should be started, get
        // a coefficient (or 0 if none available)
        if (airb_is_empty(x->inbuffer))
          x->coef2=0.0;
        else
          x->coef2=airb_get(x->inbuffer);
        // setup impulse response
        irptr  = x->iresp;
        x->ir2 = irptr;
        // set the index 2 to the value after calculation
        i=x->index1-x->P;
        x->index2=i;
        // calculate the output for grain 2
        out-=i;
        while (i--) *out++ += x->coef2 * *(irptr++);
      }
    }
  }
  else
  {
    // remark: the case irptr1=0, irptr2<>0 cannot appear
    // so we have the 3 cases that grain 1 or grain 2 changes
    // or that none of them changes
    if (x->index1+buflen>2*x->P)
    {
      // case 1: grain 1 changes
      n=2*x->P-x->index1;
      i=buflen-n;
      irptr  = x->ir1+x->index1;
      irptr2 = x->ir2+x->index2;
      while (n--) *out++ = x->coef1 * *(irptr++) + 
                           x->coef2 * *(irptr2++);
      // get a coefficient (or 0 if none available)
      if (airb_is_empty(x->inbuffer))
        x->coef1=0.0;
      else
        x->coef1=airb_get(x->inbuffer);
      // setup impulse response and index for grain 1
      irptr  = x->iresp;
      x->ir1 = irptr;
      x->index1=i;
      while (i--) *out++ = x->coef1 * *(irptr++) + 
                           x->coef2 * *(irptr2++);
      x->index2+=buflen;
    }
    else
    {
      if (x->index2+buflen>2*x->P)
      {
        // case 2: grain 2 changes
        n=2*x->P-x->index2;
        i=buflen-n;
        irptr  = x->ir1+x->index1;
        irptr2 = x->ir2+x->index2;
        while (n--) *out++ = x->coef1 * *(irptr++) + 
                             x->coef2 * *(irptr2++);
        // get a coefficient (or 0 if none available)
        if (airb_is_empty(x->inbuffer))
          x->coef2=0.0;
        else
          x->coef2=airb_get(x->inbuffer);
        // setup impulse response and index for grain 1
        irptr2 = x->iresp;
        x->ir2 = irptr2;
        x->index2=i;
        while (i--) *out++ = x->coef1 * *(irptr++) + 
                             x->coef2 * *(irptr2++);
        x->index1+=buflen;
      }
      else
      {
        // case 3: no grain changes
        irptr  = x->ir1+x->index1;
        irptr2 = x->ir2+x->index2;
        while (n--) *out++ = x->coef1 * *(irptr++) + 
                             x->coef2 * *(irptr2++);
        x->index1+=buflen;
        x->index2+=buflen;
      }
    }
  }
    
  // if in the next buffer we need a new coefficient,
  // send a bang
  // simpler: if the input buffer is empty, send a bang
  
  if (airb_is_empty(x->inbuffer))
    outlet_bang(x->b_out);
  
  return (w+4);
}

void modgrain_tilde_dsp(t_modgrain_tilde *x, t_signal **sp)
{
  dsp_add(modgrain_tilde_perform, 3, x,
          sp[0]->s_vec,sp[0]->s_n);
}

// constructor
void *modgrain_tilde_new(t_floatarg P, t_floatarg k)
{
  t_int        irlen,n,k_int,P_int;
  t_sample     *irptr;
  t_float      pi_div_2P;
  t_modgrain_tilde *x = (t_modgrain_tilde *)pd_new(modgrain_tilde_class);

  // set P,k from constructor arguments
  P_int = (t_int)P;
  if (P_int<P_MIN)
    P_int=P_MIN;
  if (P_int>P_MAX)
    P_int=P_MAX;

  k_int = (t_int)k;
  
  x->P = P_int;
  x->k = k_int;
  
  x->alpha = 0;
  x->gamma = 0;
  
  x->ir1  = 0;
  x->ir2  = 0;
  
  // set up impulse response
  irlen    = 2*x->P;
  irptr    = getbytes(irlen*sizeof(t_sample));
  x->iresp = irptr;
  x->iresp2 = getbytes(irlen*sizeof(t_sample));
  x->iresp3 = getbytes(irlen*sizeof(t_sample));
  n=0;
  pi_div_2P=M_PI/(2.0*P_int);
  while (n<irlen) {
    *irptr++ = sinf(n*pi_div_2P)*
               cosf((2*n+P_int+1)*(2*k_int+1)*0.5*pi_div_2P);
    n++;
  }
  
  // create input buffer
  x->inbuffer=rb_create(3);
  
  // create new inlets for P and k
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("P"));
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("k"));
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("ampwarp"));
  inlet_new(&x->x_obj, &x->x_obj.ob_pd,
            &s_float, gensym("timewarp"));
  
  // signal outlet
  outlet_new(&x->x_obj, &s_signal);
  // coefficient request outlet
  x->b_out = outlet_new(&x->x_obj, &s_bang);  
  
  return (void *)x;
}

// destructor
void modgrain_tilde_free(t_modgrain_tilde *x)
{
  rb_destroy(x->inbuffer);
  freebytes(x->iresp,2*x->P*sizeof(t_sample));
}

void modgrain_tilde_setup(void) {
  modgrain_tilde_class = class_new(gensym("modgrain~"),
        (t_newmethod)modgrain_tilde_new,
	(t_method)modgrain_tilde_free,
        sizeof(t_modgrain_tilde),
	CLASS_DEFAULT,
	A_DEFFLOAT, A_DEFFLOAT, 0);
  
  class_addfloat(modgrain_tilde_class, modgrain_tilde_coef);
  class_addmethod(modgrain_tilde_class, (t_method)modgrain_tilde_P,
                  gensym("P"), A_DEFFLOAT, 0);
  class_addmethod(modgrain_tilde_class, (t_method)modgrain_tilde_k,
                  gensym("k"), A_DEFFLOAT, 0);
  class_addmethod(modgrain_tilde_class, (t_method)modgrain_tilde_ampwarp,
                  gensym("ampwarp"), A_DEFFLOAT, 0);
  class_addmethod(modgrain_tilde_class, (t_method)modgrain_tilde_timewarp,
                  gensym("timewarp"), A_DEFFLOAT, 0);
  class_addmethod(modgrain_tilde_class,
        (t_method)modgrain_tilde_dsp, gensym("dsp"), 0);
}
