// Fractal Additive Synthesis Partial Control
// (inharmonic version)

// Input : Parameters (start, lpcnorm, SIGLEN),
//         P, buffer underrun (bang)
// Output: Time for the three scales,
//         coefficients (left and right)

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangeli sta]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.


#include <stdlib.h>
#include "m_pd.h"

#define RM_2 (RAND_MAX/2)

static t_class *pcontrol12_class;

typedef struct _pcontrol12 {
  t_object x_obj;
  
  t_outlet *t3out, *t2out, *t1out;
  t_outlet *cl3out,*cl2out,*cl1out;
  t_outlet *cr3out,*cr2out,*cr1out;
  
  long int siglen, P, time;
  // LPC norms
  t_float  l0n,l1n,l2n,r0n,r1n,r2n;
} t_pcontrol12;

void pcontrol12_sendcoef(t_pcontrol12 *x)
{
  t_float  t,siglen;
  long int time;
  
  // Output coefficients only when the sound
  // is still active (this keeps CPU usage very
  // low when the sound is finished)
  if (x->time<x->siglen) {
    time   = x->time;
    siglen = x->siglen;
    // t: time/siglen
    t=(t_float)time/siglen;
    // first scale
    outlet_float(x->t1out,t);
    // send random, normalised float (in [-1,1])
    outlet_float(x->cl1out,x->l0n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr1out,x->r0n*(rand()-RM_2)/(t_float)RM_2);
    // second scale
    outlet_float(x->t2out,t);
    outlet_float(x->cl2out,x->l1n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr2out,x->r1n*(rand()-RM_2)/(t_float)RM_2);
    // third scale
    outlet_float(x->t3out,t);
    outlet_float(x->cl3out,x->l2n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr3out,x->r2n*(rand()-RM_2)/(t_float)RM_2);
    // t: (time+2*P)/siglen
    time += 2*x->P;
    t=(t_float)time/siglen;
    // first scale
    outlet_float(x->t1out,t);
    outlet_float(x->cl1out,x->l0n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr1out,x->r0n*(rand()-RM_2)/(t_float)RM_2);
    // t: (time+4*P)/siglen
    time += 2*x->P;
    t=(t_float)time/siglen;
    // first scale
    outlet_float(x->t1out,t);
    outlet_float(x->cl1out,x->l0n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr1out,x->r0n*(rand()-RM_2)/(t_float)RM_2);
    // second scale
    outlet_float(x->t2out,t);
    outlet_float(x->cl2out,x->l1n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr2out,x->r1n*(rand()-RM_2)/(t_float)RM_2);
    // t: (time+6*P)/siglen
    time += 2*x->P;
    t=(t_float)time/siglen;
    // first scale
    outlet_float(x->t1out,t);
    outlet_float(x->cl1out,x->l0n*(rand()-RM_2)/(t_float)RM_2);
    outlet_float(x->cr1out,x->r0n*(rand()-RM_2)/(t_float)RM_2);
    // write back x->time+8*P
    x->time=time+2*x->P;
  }
}

void pcontrol12_start(t_pcontrol12 *x)
{
  x->time = 0;
  // output eight times 8 coefficients because we drop 69 of them
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
  pcontrol12_sendcoef(x);
}

void pcontrol12_P(t_pcontrol12 *x, t_floatarg f)
{
  long int P_int=f;
  
  if (P_int<=0)
    P_int=1;
  
  x->P=P_int;
}

void pcontrol12_SIGLEN(t_pcontrol12 *x, t_floatarg f)
{
  long int siglen_int=f;
  
  if (siglen_int<=0)
    siglen_int=1;
  
  x->siglen=siglen_int;
}

void pcontrol12_l0lpcnorm(t_pcontrol12 *x, t_floatarg f)
{
  x->l0n=f;
}

void pcontrol12_l1lpcnorm(t_pcontrol12 *x, t_floatarg f)
{
  x->l1n=f;
}

void pcontrol12_l2lpcnorm(t_pcontrol12 *x, t_floatarg f)
{
  x->l2n=f;
}

void pcontrol12_r0lpcnorm(t_pcontrol12 *x, t_floatarg f)
{
  x->r0n=f;
}

void pcontrol12_r1lpcnorm(t_pcontrol12 *x, t_floatarg f)
{
  x->r1n=f;
}

void pcontrol12_r2lpcnorm(t_pcontrol12 *x, t_floatarg f)
{
  x->r2n=f;
}

// constructor
void *pcontrol12_new(void)
{
  t_pcontrol12 *x=(t_pcontrol12 *)pd_new(pcontrol12_class);
  
  inlet_new(&x->x_obj, &x->x_obj.ob_pd, &s_float, gensym("P"));
  inlet_new(&x->x_obj, &x->x_obj.ob_pd, &s_bang, gensym("sendcoef"));
  
  x->t3out  = outlet_new(&x->x_obj, &s_float);
  x->t2out  = outlet_new(&x->x_obj, &s_float);
  x->t1out  = outlet_new(&x->x_obj, &s_float);
  x->cl3out = outlet_new(&x->x_obj, &s_float);
  x->cl2out = outlet_new(&x->x_obj, &s_float);
  x->cl1out = outlet_new(&x->x_obj, &s_float);
  x->cr3out = outlet_new(&x->x_obj, &s_float);
  x->cr2out = outlet_new(&x->x_obj, &s_float);
  x->cr1out = outlet_new(&x->x_obj, &s_float);
  
  // set default data values
  x->siglen = 1;
  x->time   = 0;
  x->P      = 64;
  x->l0n    = 1;
  x->l1n    = 1;
  x->l2n    = 1;
  x->r0n    = 1;
  x->r1n    = 1;
  x->r2n    = 1;
  
  return (void *)x;
}

void pcontrol12_setup(void) {
  pcontrol12_class = class_new(gensym("pcontrol12"),
                         (t_newmethod)pcontrol12_new,
			 0, sizeof(t_pcontrol12),
			 CLASS_DEFAULT, 0);
  
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_sendcoef,
                  gensym("sendcoef"),0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_start,
                  gensym("start"),0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_P,
                  gensym("P"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_SIGLEN,
                  gensym("SIGLEN"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_l0lpcnorm,
                  gensym("l0lpcnorm"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_l1lpcnorm,
                  gensym("l1lpcnorm"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_l2lpcnorm,
                  gensym("l2lpcnorm"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_r0lpcnorm,
                  gensym("r0lpcnorm"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_r1lpcnorm,
                  gensym("r1lpcnorm"),A_DEFFLOAT,0);
  class_addmethod(pcontrol12_class, (t_method)pcontrol12_r2lpcnorm,
                  gensym("r2lpcnorm"),A_DEFFLOAT,0);
}

