// Ringbuffer for floating-point numbers
// using only PD data types

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangeli sta]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.


// Two types of buffers are implemented:
//   Auto-increase (prefix: airb_):
//     Dynamically increases the buffer length as needed,
//     Methods: put, get
//   Fixed-length (function prefix: flrb_):
//     length of buffer is fix,
//     write pointer serves also as read pointer
//     Methods: put, get, conv (convolution), mult_add

#include "pd_ringbuffer.h"

// creates a new ringbuffer for size numbers
t_rb_info *rb_create(size_t size) {
  t_rb_info *info;
  
  info = getbytes(sizeof(t_rb_info));
  info->start     = getbytes(size*sizeof(t_float));
  info->write_ptr = 0;
  info->read_ptr  = 0;
  info->is_empty  = 1;
  info->size      = size;
  
  return info;
}

// frees all memory occupied by the buffer
// (including info field)
void rb_destroy(t_rb_info *info) {
  freebytes(info->start,info->size*sizeof(t_float));
  freebytes(info,sizeof(t_rb_info));
}

// initialises to 0 the whole buffer
void flrb_init(t_rb_info *info) {
  t_int   i;
  t_float *buffer;

  buffer = info->start;
  info->write_ptr = 0;
  info->read_ptr  = 0;
  
  for (i=0;i<info->size;i++)
    buffer[i]=0.0;
}

// returns 1 of buffer is empty, 0 otherwise
inline t_int airb_is_empty(t_rb_info *info) {
  return (info->is_empty);
}

// puts a floating-point number into the buffer
// (overwriting whatever was there before)
inline void flrb_put(t_rb_info *info, t_float f) {
  t_float *buffer = info->start;
  
  buffer[info->write_ptr++]=f;
  if (info->write_ptr>=info->size)
    info->write_ptr=0;
}

// gets a floating-point number from the buffer
// (not moving the write pointer)
inline t_float flrb_get(t_rb_info *info) {
  return info->start[info->write_ptr];
}

// calculates the convolution between the buffer content
// and iresp (supposed to have the same length as the buffer)
// ATTENTION: iresp is the time-reversed impulse response
inline t_float flrb_conv(t_rb_info *info, t_float *iresp) {
  t_int   i;
  t_float to_return = 0;
  t_float *buffer   = info->start;
  
  // iresp : 0         -> size-write_ptr-1
  // buffer: write_ptr -> size-1
  for (i=0; i < info->size - info->write_ptr; i++)
    to_return+=buffer[i+info->write_ptr]*iresp[i];
  // iresp : size-write_ptr -> size-1
  // buffer: 0              -> write_ptr-1
  for (i=0; i < info->write_ptr; i++)
    to_return+=buffer[i]*iresp[i+info->size-info->write_ptr];

  return to_return;
}

// adds the impulse response iresp, multiplied by coef, to
// the buffer
inline void flrb_mult_add(t_rb_info *info, t_float *iresp,
                             t_float coef) {
  t_int   i;
  t_float *buffer   = info->start;
  
  // iresp : 0         -> size-write_ptr-1
  // buffer: write_ptr -> size-1
  for (i=0; i < info->size - info->write_ptr; i++)
    buffer[i+info->write_ptr]+=coef*iresp[i];
  // iresp : size-write_ptr -> size-1
  // buffer: 0              -> write_ptr-1
  for (i=0; i < info->write_ptr; i++)
    buffer[i]+=coef*iresp[i+info->size-info->write_ptr];
}

// puts a float into the buffer specified by info
// and automatically increases the buffer size
// in order to avoid buffer overflows
inline void airb_put(t_rb_info *info, t_float f) {
  t_int   i;
  t_float *buffer;
  
  // if the buffer is full ...
  if (info->read_ptr == info->write_ptr && info->is_empty==0) {
    // ... create new buffer of double size ...
    buffer=getbytes(2 * info->size * sizeof(t_float));
    post("Increasing buffer size to %d", 2*info->size);
    
    // ... copy the data in it ...
    for (i=info->read_ptr; i<info->size; i++)
      buffer[i-info->read_ptr]=info->start[i];
    for (i=0; i<info->write_ptr; i++)
      buffer[i+info->size-info->write_ptr]=info->start[i];
    
    // ... destroy the old buffer ...
    freebytes(info->start,info->size*sizeof(t_float));
    
    // ... link the new buffer to the info field ...
    info->start=buffer;
    
    // ... adjust the read and write pointers ...
    info->read_ptr  = 0;
    info->write_ptr = info->size;
    
    // ... and finally enter the new buffer size
    info->size = 2 * info->size;
  }
  
  // now the buffer is certainly ready to accomodate the
  // new data
  info->start[info->write_ptr++]=f;
  
  if (info->write_ptr >= info->size)
    info->write_ptr = 0;
  
  info->is_empty = 0;
}

// gets a float from the buffer specified by info
// ATTENTION: No error-checking for buffer underruns
// (use airb_is_empty before calling this method)
inline t_float airb_get(t_rb_info *info) {
  t_float f;
  
  f=info->start[info->read_ptr++];
  if (info->read_ptr >= info->size)
    info->read_ptr = 0;
  
  if (info->read_ptr == info->write_ptr)
    info->is_empty = 1;
  
  return f;
}
