// Fractal Additive Synthesis Voice Control
// (inharmonic version)

// Input : midi pitch (takes also "config" message), amplitude
// Output: parameters

// This File is part of the 
// Fractal Additive Synthesis Tools for Pure Data

// Copyright (C) 2002, Fritz Menzer
// Ecole Polytechnique Fdrale de Lausanne
// (EPFL)

// Fractal Additive Synthesis Technology:
// Copyright (C) Pietro Polotti and Gianpaolo Evangelista
// Laboratoire de Communications Audiovisuelles
// (LCAV)
// cole Polytechnique Fdrale de Lausanne
// DSC-INR, Ecublens
// CH-1015, Lausanne, Switzerland
// [pietro.polotti,gianpaolo.evangeli sta]@epfl.ch

// You are free to use this code for any non-commercial purpose,
// provided that the above copyright notices are included in any
// software, hardware or publication derived from this code.


#include <stdio.h>
#include <string.h>
#include "m_pd.h"

// header strings for single-pitch
// and multi-pitch sounds
#define SP_HEADER "fassffsp"
#define MP_HEADER "fassffmp"
#define WT_SCALES 3

static t_class *vcontrol_class;

typedef struct _scale {
  t_int   lpclen;
  t_float *lpccoef;
  t_float lpcnorm;
  
  t_int   amplen;
  t_float *ampenv;
} t_scale;

typedef struct _partial {
  t_int   P,k;
  t_float IrIl;
  // sinusoidal part
  t_int   sinamplen;
  t_float *sinampenv;
  t_int   sinphaselen;
  t_float *sinphaseenv;
  // stochastic part
  t_scale l[3],r[3];
} t_partial;

typedef struct _vcontrol {
  t_object  x_obj;
  
  t_int     hwpartnum,partnum;
  long int  siglen;
  t_partial *partials;
  
  t_float   amplitude;
  
  t_outlet  *route_out,*all_out;
  
  t_int     configured;
  t_int     partials_ok;
} t_vcontrol;

void vcontrol_config(t_vcontrol *x)
{
  t_int  i,j,k;
  t_atom *message,oneatom;
  char wtselector[30];
  // send the sample length to all partials
  SETFLOAT(&oneatom, (t_float)x->siglen); // signal length
  outlet_anything(x->all_out,gensym("SIGLEN"),1,&oneatom);
  // send all the partials data to the outlets
  for (i=0; i<x->partnum; i++) {
    // send all the scalars
    message=getbytes(4*sizeof(t_atom));
    SETFLOAT(&message[0], (t_float)i);    // partial number
    // P
    SETSYMBOL(&message[1], gensym("P"));  // P identifier
    SETFLOAT(&message[2], (t_float)(x->partials[i].P));
    outlet_list(x->route_out,&s_list,3,message);
    // k
    SETSYMBOL(&message[1], gensym("k"));  // k identifier
    SETFLOAT(&message[2], (t_float)(x->partials[i].k));
    outlet_list(x->route_out,&s_list,3,message);
    // Ir/Il
    SETSYMBOL(&message[1], gensym("IrIl")); // identifier
    SETFLOAT(&message[2], (t_float)(x->partials[i].IrIl));
    outlet_list(x->route_out,&s_list,3,message);
    // sinamplen
    SETSYMBOL(&message[1], gensym("sinamp")); // ident
    SETSYMBOL(&message[2], gensym("resize")); // resize
    SETFLOAT(&message[3], (t_float)(x->partials[i].sinamplen));
    outlet_list(x->route_out,&s_list,4,message);
    // sinphaselen
    SETSYMBOL(&message[1], gensym("sinphase")); // ident
    SETSYMBOL(&message[2], gensym("resize")); // resize
    SETFLOAT(&message[3], (t_float)(x->partials[i].sinphaselen));
    outlet_list(x->route_out,&s_list,4,message);
    // wavelet scales
    for (k=0;k<3;k++) {
      // left: LPC length, LPC norm, amp length
      sprintf(wtselector,"l%d",k); // generate start of selector
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"lpcnorm"))); 
      SETFLOAT(&message[2], (t_float)(x->partials[i].l[k].lpcnorm));
      outlet_list(x->route_out,&s_list,3,message);
      sprintf(wtselector,"l%d",k); // generate start of selector
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"amp"))); 
      SETSYMBOL(&message[2], gensym("resize")); // resize
      SETFLOAT(&message[3], (t_float)(x->partials[i].l[k].amplen));
      outlet_list(x->route_out,&s_list,4,message);
      // right: LPC length, LPC norm, amp length
      sprintf(wtselector,"r%d",k); // generate start of selector
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"lpcnorm"))); 
      SETFLOAT(&message[2], (t_float)(x->partials[i].r[k].lpcnorm));
      outlet_list(x->route_out,&s_list,3,message);
      sprintf(wtselector,"r%d",k); // generate start of selector
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"amp"))); 
      SETSYMBOL(&message[2], gensym("resize")); // resize
      SETFLOAT(&message[3], (t_float)(x->partials[i].r[k].amplen));
      outlet_list(x->route_out,&s_list,4,message);
    }
    freebytes(message,4*sizeof(t_atom));

    // send all the lists
    // sinampenv
    message=getbytes((3+x->partials[i].sinamplen)*sizeof(t_atom));
    SETFLOAT(&message[0], (t_float)i);    // partial number
    SETSYMBOL(&message[1], gensym("sinamp"));
    SETFLOAT(&message[2], 0);    // start from index 0
    for (j=0;j<x->partials[i].sinamplen;j++)
      SETFLOAT(&message[j+3], (t_float)(x->partials[i].sinampenv[j]));
    outlet_list(x->route_out,&s_list,3+x->partials[i].sinamplen,message);
    freebytes(message,(3+x->partials[i].sinamplen)*sizeof(t_atom));
    // sinphaseenv
    message=getbytes((3+x->partials[i].sinphaselen)*sizeof(t_atom));
    SETFLOAT(&message[0], (t_float)i);    // partial number
    SETSYMBOL(&message[1], gensym("sinphase"));
    SETFLOAT(&message[2], 0);    // start from index 0
    for (j=0;j<x->partials[i].sinphaselen;j++)
      SETFLOAT(&message[j+3], (t_float)(x->partials[i].sinphaseenv[j]));
    outlet_list(x->route_out,&s_list,3+x->partials[i].sinphaselen,message);
    freebytes(message,(3+x->partials[i].sinphaselen)*sizeof(t_atom));
    // wavelet scales
    for (k=0;k<3;k++) {
      // left: LPC coef., amp. env.
      sprintf(wtselector,"l%d",k); // generate start of selector
      message=getbytes((2+x->partials[i].l[k].lpclen)*sizeof(t_atom));
      SETFLOAT(&message[0], (t_float)i);    // partial number
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"lpccoef")));
      for (j=0;j<x->partials[i].l[k].lpclen;j++)
        SETFLOAT(&message[j+2], (t_float)(x->partials[i].l[k].lpccoef[j]));
      outlet_list(x->route_out,&s_list,2+x->partials[i].l[k].lpclen,message);
      freebytes(message,(2+x->partials[i].l[k].lpclen)*sizeof(t_atom));

      sprintf(wtselector,"l%d",k); // generate start of selector
      message=getbytes((3+x->partials[i].l[k].amplen)*sizeof(t_atom));
      SETFLOAT(&message[0], (t_float)i);    // partial number
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"amp")));
      SETFLOAT(&message[2], 0);    // start from index 0
      for (j=0;j<x->partials[i].l[k].amplen;j++)
        SETFLOAT(&message[j+3], (t_float)(x->partials[i].l[k].ampenv[j]));
      outlet_list(x->route_out,&s_list,3+x->partials[i].l[k].amplen,message);
      freebytes(message,(3+x->partials[i].l[k].amplen)*sizeof(t_atom));

      // right: LPC coef., amp. env.
      sprintf(wtselector,"r%d",k); // generate start of selector
      message=getbytes((2+x->partials[i].r[k].lpclen)*sizeof(t_atom));
      SETFLOAT(&message[0], (t_float)i);    // partial number
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"lpccoef")));
      for (j=0;j<x->partials[i].r[k].lpclen;j++)
        SETFLOAT(&message[j+2], (t_float)(x->partials[i].r[k].lpccoef[j]));
      outlet_list(x->route_out,&s_list,2+x->partials[i].r[k].lpclen,message);
      freebytes(message,(2+x->partials[i].r[k].lpclen)*sizeof(t_atom));

      sprintf(wtselector,"r%d",k); // generate start of selector
      message=getbytes((3+x->partials[i].r[k].amplen)*sizeof(t_atom));
      SETFLOAT(&message[0], (t_float)i);    // partial number
      SETSYMBOL(&message[1], gensym(strcat(wtselector,"amp")));
      SETFLOAT(&message[2], 0);    // start from index 0
      for (j=0;j<x->partials[i].r[k].amplen;j++)
        SETFLOAT(&message[j+3], (t_float)(x->partials[i].r[k].ampenv[j]));
      outlet_list(x->route_out,&s_list,3+x->partials[i].r[k].amplen,message);
      freebytes(message,(3+x->partials[i].r[k].amplen)*sizeof(t_atom));
    }
  }
  x->configured = 1;
}

void vcontrol_midi(t_vcontrol *x, t_floatarg f)
{
  t_atom oneatom;
  // check if configured
  if (x->configured==0)
    vcontrol_config(x);
  // send the sample length to all partials
  SETFLOAT(&oneatom, (t_float)x->siglen); // signal length
  outlet_anything(x->all_out,gensym("SIGLEN"),1,&oneatom);
  // send a start message to all channels
  outlet_anything(x->all_out, gensym("start"),0,0);
}

// constructor:
// read all the data from the file
// and store it partial by partial
void *vcontrol_new(t_symbol *datafile, t_floatarg hwpartnum)
{
  t_vcontrol *x=(t_vcontrol *)pd_new(vcontrol_class);
  FILE       *fp;
  char       header[9];
  short int  partnum,i,j,k;
  
  floatinlet_new(&x->x_obj, &x->amplitude);
  
  x->all_out   = outlet_new(&x->x_obj, 0);
  x->route_out = outlet_new(&x->x_obj, &s_list);
  
  // set default data values
  x->configured  = 0;
  x->partials_ok = 0;
  x->hwpartnum = (t_int)hwpartnum;
  
  if ((fp = fopen(datafile->s_name, "rb"))==NULL) {
    post("Error: File %s does not exist",datafile->s_name);
    return (void *)x;
  }
  
  // check header
  fread(header,1,8,fp);
  header[8]=0;
  
  if (strcmp(header,SP_HEADER)) {
    post("Error: File %s is not a FAS soundfile",datafile->s_name);
    fclose(fp);
    return (void *)x;
  }
  
  fread(&partnum,2,1,fp);
  
  // if hwpartnum undefined or greater than # of partials in file,
  // use all of the partials in file, else restrict to hwpartnum
  if (!((partnum < x->hwpartnum) || (x->hwpartnum==0)))
    partnum=x->hwpartnum;
  
  x->partnum=partnum;
  
  // check that the number of WT scales is three
  fread(&i,2,1,fp);
  
  if (i!=3) {
    post("Error: File %s contains %d!=3 WT scales",
         datafile->s_name,i);
    fclose(fp);
    return (void *)x;
  }
  
  // read WT filter length into i (temporary)
  fread(&i,2,1,fp);
  
  // read signal length
  fread(&x->siglen,4,1,fp);
  
  // skip the WT filters
  fseek(fp,i*2*sizeof(t_float),SEEK_CUR);
  
  // allocate memory for the partials' informations
  x->partials=getbytes(partnum*sizeof(t_partial));
  
  for (i=0; i<partnum; i++) {
    // read P,k
    fread(&x->partials[i].P,2,1,fp);
    fread(&x->partials[i].k,2,1,fp);
    post("Partial %d: P=%d, k=%d",i,x->partials[i].P,x->partials[i].k);
    
    // read Il/Ir
    fread(&x->partials[i].IrIl,sizeof(t_float),1,fp);
    
    // read sinamp
    fread(&j,2,1,fp);
    x->partials[i].sinamplen=j;
    x->partials[i].sinampenv=getbytes(j*sizeof(t_float));
    fread(x->partials[i].sinampenv,sizeof(t_float),j,fp);
    
    // read sinphase
    fread(&j,2,1,fp);
    x->partials[i].sinphaselen=j;
    x->partials[i].sinphaseenv=getbytes(j*sizeof(t_float));
    fread(x->partials[i].sinphaseenv,sizeof(t_float),j,fp);
    
    // read lpc coefs
    for (k=0;k<3;k++) {
      // left coefs and norm. factor
      fread(&j,2,1,fp);
      x->partials[i].l[k].lpclen=j;
      x->partials[i].l[k].lpccoef=getbytes(j*sizeof(t_float));
      fread(x->partials[i].l[k].lpccoef,sizeof(t_float),j,fp);
      fread(&x->partials[i].l[k].lpcnorm,sizeof(t_float),1,fp);
      // left amplitude envelope
      fread(&j,2,1,fp);
      x->partials[i].l[k].amplen=j;
      x->partials[i].l[k].ampenv=getbytes(j*sizeof(t_float));
      fread(x->partials[i].l[k].ampenv,sizeof(t_float),j,fp);
      // right coefs and norm. factor
      fread(&j,2,1,fp);
      x->partials[i].r[k].lpclen=j;
      x->partials[i].r[k].lpccoef=getbytes(j*sizeof(t_float));
      fread(x->partials[i].r[k].lpccoef,sizeof(t_float),j,fp);
      fread(&x->partials[i].r[k].lpcnorm,sizeof(t_float),1,fp);
      // right amplitude envelope
      fread(&j,2,1,fp);
      x->partials[i].r[k].amplen=j;
      x->partials[i].r[k].ampenv=getbytes(j*sizeof(t_float));
      fread(x->partials[i].r[k].ampenv,sizeof(t_float),j,fp);
    }
  }
  
  fclose(fp);
  x->partials_ok = 1;
  return (void *)x;
}

// destructor
void vcontrol_free(t_vcontrol *x) {
  t_int i,k;
  
  if (x->partials_ok) {
    for (i=0;i<x->partnum;i++) {
      // free the memory for the LPC coefs
      for (k=0;k<3;k++) {
        freebytes(x->partials[i].l[k].lpccoef,
                  x->partials[i].l[k].lpclen*sizeof(t_float));
        freebytes(x->partials[i].l[k].ampenv,
                  x->partials[i].l[k].amplen*sizeof(t_float));
        freebytes(x->partials[i].r[k].lpccoef,
                  x->partials[i].r[k].lpclen*sizeof(t_float));
        freebytes(x->partials[i].r[k].ampenv,
                  x->partials[i].r[k].amplen*sizeof(t_float));
      }
      // free the memory for the sinusoidal part
      freebytes(x->partials[i].sinampenv,
                x->partials[i].sinamplen*sizeof(t_float));
      freebytes(x->partials[i].sinphaseenv,
                x->partials[i].sinphaselen*sizeof(t_float));
    }
    // free the memory for the partial information
    // (pointers, length, etc...)
    freebytes(x->partials,x->partnum*sizeof(t_partial));
  }
}

void vcontrol_setup(void) {
  vcontrol_class = class_new(gensym("vcontrol"),
                         (t_newmethod)vcontrol_new,
			 (t_method)vcontrol_free,
                         sizeof(t_vcontrol),
			 CLASS_DEFAULT,
                         A_DEFSYMBOL, A_DEFFLOAT,0);
  class_addfloat(vcontrol_class, vcontrol_midi);
  class_addmethod(vcontrol_class, (t_method)vcontrol_config,
                  gensym("config"),0);
}

